/* -*-c++-*- Copyright (C) 2018 Advanced Driver Information Technology.
 *
 * This library is open source and may be redistributed and/or modified under
 * the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or
 * (at your option) any later version.  The full license is in LICENSE file
 * included with this distribution, and on the openscenegraph.org website.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * OpenSceneGraph Public License for more details.
*/

#include "WLShell.hpp"

#include <errno.h>
#include <string>

#include "WLSurfaceEvent.hpp"

using namespace WaylandBackend;

const struct wl_shell_surface_listener WLShell::surface_listener = {
    WLShell::handleSurfacePing,
    WLShell::handleSurfaceConfigure,
    WLShell::handleSurfacePopupDone
};


WLShell::WLShell(struct wl_event_queue *eventQ)
:_wlEventQ(eventQ)
,_wlID(0)
,_wlInterfaceName("wl_shell")
,_wlShell(NULL)
,_shellName("WLShell")
{

}

int
WLShell::init(struct wl_registry* registry, uint32_t name,
              const char* interface, uint32_t version)
{
    int ret = -EINVAL;
    std::string ifName(interface);

    if (ifName == _wlInterfaceName)
    {
        _wlID = name;

        ret = 0;

        if (NULL != _wlEventQ)
        {
            struct wl_registry* wrappedRegistry = (struct wl_registry*)
                             wl_proxy_create_wrapper((void *)registry);
            if (NULL != wrappedRegistry)
            {
                wl_proxy_set_queue((struct wl_proxy*)wrappedRegistry,
                                   _wlEventQ);
                _wlShell = (struct wl_shell*)wl_registry_bind(registry, name,
                        &wl_shell_interface, version);
                wl_proxy_wrapper_destroy(wrappedRegistry);
            }
        }
        else
        {
            /*
            * It is not good idea to use default queue. default queue can be
            * dispatched by undesired threads
            */
            _wlShell = (struct wl_shell *)wl_registry_bind(registry, name,
                &wl_shell_interface, version);
        }
        if (NULL == _wlShell)
        {
            ret = -ENOMEM;
        }
    }
    return ret;
}

void
WLShell::handleSurfaceConfigure(void *data,
                                struct wl_shell_surface *wlShellSurface,
                                uint32_t edges,
                                int32_t width,
                                int32_t height)
{
    WLSurfaceEvent *surfEvent;

    surfEvent = static_cast<WLSurfaceEvent *>
                     (wl_shell_surface_get_user_data(wlShellSurface));
    if (NULL != surfEvent)
        surfEvent->surfaceConfigured(edges, width, height);
}

void
WLShell::handleSurfacePing(void *data,
                     struct wl_shell_surface *wl_shell_surface,
                     uint32_t serial)
{
    wl_shell_surface_pong(wl_shell_surface, serial);
}

void
WLShell::handleSurfacePopupDone(void *data,
           struct wl_shell_surface *wl_shell_surface)
{
    //TODO: As of now there is no use case to handle it. further investigate
    // possibility in future.
}


int
WLShell::surfaceCreate(struct wl_surface* wlSurface,
                       unsigned int surfaceId,
                       void **shellSurface)
{
    int ret = -EINVAL;

    if (NULL != shellSurface)
    {
        *shellSurface = (void *)wl_shell_get_shell_surface(_wlShell,
                                wlSurface);
        if (NULL != *shellSurface)
        {
            wl_shell_surface_add_listener((struct wl_shell_surface *)
                                          *shellSurface,
                                          &surface_listener, this);

            wl_shell_surface_set_user_data((struct wl_shell_surface*)
                                           *shellSurface,
                                          wl_surface_get_user_data(wlSurface));

            surfaceSetTopLevel(*shellSurface);
            ret = 0;
        }
    }
    return ret;
}

int
WLShell::surfaceDestroy(void* shellSurface)
{
    int ret = -EINVAL;
    if (NULL != shellSurface)
    {
        wl_shell_surface_destroy((struct wl_shell_surface*)shellSurface);
        ret = 0;
    }
    return ret;
}

int
WLShell::surfaceSetTitle(void* shellSurface, const char *title)
{
    int ret = -EINVAL;
    if ((NULL != shellSurface) && (NULL != title))
    {
        wl_shell_surface_set_title((struct wl_shell_surface*)shellSurface,
                                   title);
    }
    return ret;
}

int
WLShell::surfaceSetClass(void* shellSurface, const char *surfClass)
{
    int ret = -EINVAL;
    if (NULL != shellSurface)
    {
        wl_shell_surface_set_class((struct wl_shell_surface*)shellSurface,
                                   surfClass);
    }
    return ret;
}

int
WLShell::surfaceSetAsPopup(void* shellSurface, struct wl_seat *seat,
                           uint32_t serial, struct wl_surface *parent,
                           int32_t x, int32_t y, uint32_t flags)
{
    int ret = -EINVAL;
    if ((NULL != shellSurface) && (NULL != seat) && (NULL != parent))
    {
        wl_shell_surface_set_popup((struct wl_shell_surface *) shellSurface,
                                   seat, serial, parent, x , y, flags);
        ret = 0;
    }
    return ret;
}

int
WLShell::surfaceSetMaximized(void* shellSurface, struct wl_output *output)
{
    int ret = -EINVAL;
    if ((NULL != shellSurface) && (NULL != output))
    {
        wl_shell_surface_set_maximized((struct wl_shell_surface *)shellSurface,
                output);
        ret = 0;
    }
    return ret;
}

int
WLShell::surfaceSetFullScreen(void *shellSurface, uint32_t method,
                              uint32_t framerate, struct wl_output *output)
{
    int ret = -EINVAL;
    if ((NULL != shellSurface) && (NULL != output))
    {
        wl_shell_surface_set_fullscreen((struct wl_shell_surface *)
                                        shellSurface, method,
                                        framerate, output);
        ret = 0;
    }
    return ret;
}

int
WLShell::surfaceSetTransient(void *shellSurface, struct wl_surface *parent,
                             int32_t x, int32_t y, uint32_t flags)
{
    int ret = -EINVAL;
    if (NULL != shellSurface)
    {
        wl_shell_surface_set_transient((struct wl_shell_surface *)shellSurface,
                                       parent, x, y, flags);
        ret = 0;
    }
    return ret;
}

int
WLShell::surfaceSetTopLevel(void *shellSurface)
{
    int ret = -EINVAL;
    if (NULL != shellSurface)
    {
        wl_shell_surface_set_toplevel((struct wl_shell_surface *)shellSurface);
        ret = 0;
    }
    return ret;
}

int
WLShell::surfaceResize(void *shellSurface, struct wl_seat *seat,
                       uint32_t serial, uint32_t edges)
{
    int ret = -EINVAL;
    if ((NULL != shellSurface) && (NULL != seat))
    {
        wl_shell_surface_resize((struct wl_shell_surface *) shellSurface,
                                 seat, serial, edges);
        ret= 0;
    }
    return ret;
}

int
WLShell::surfaceMove(void *shellSurface, struct wl_seat *seat, uint32_t serial)
{
    int ret = -EINVAL;
    if ((NULL != shellSurface) && (NULL != seat))
    {
        wl_shell_surface_move((struct wl_shell_surface *) shellSurface, seat,
                              serial);
        ret= 0;
    }
    return ret;
}

uint32_t
WLShell::getWLID()
{
    return _wlID;
}

std::string&
WLShell::getWLName()
{
    return _wlInterfaceName;
}

WLShell::~WLShell()
{
    if (NULL != _wlShell)
    {
        wl_shell_destroy(_wlShell);
    }
}


